<?php
namespace App\Controllers;

use App\Config\Config;
use App\Helpers\Logger;
use App\Helpers\Keyboard;
use App\Models\Database;
use App\Controllers\AdminController;
use PDO;
use Exception;
use App\Helpers\V2RayPanel;

class BotController {
    private $update;
    private $chatId;
    private $text;
    private $isAdmin;
    private $db;
    private $firstName;
    private $username;
    private $messageId; // برای ویرایش پیام‌ها
    private $isMainAdmin;

    public function __construct($update) {
        $this->update = $update;
        $this->db = Database::getInstance();
        
        // پردازش ورودی (پیام متنی یا دکمه شیشه‌ای)
        if (isset($update['callback_query'])) {
            $this->chatId = $update['callback_query']['message']['chat']['id'];
            $this->text = $update['callback_query']['data'];
            $this->firstName = $update['callback_query']['message']['chat']['first_name'] ?? 'User';
            $this->username = $update['callback_query']['message']['chat']['username'] ?? '';
            $this->messageId = $update['callback_query']['message']['message_id'];
        } else {
            $this->chatId = $update['message']['chat']['id'] ?? null;
            $this->text = $update['message']['text'] ?? null;
            $this->firstName = $update['message']['chat']['first_name'] ?? 'User';
            $this->username = $update['message']['chat']['username'] ?? '';
            $this->messageId = $update['message']['message_id'] ?? null;
        }

// بررسی ادمین اصلی (داخل فایل Config)
        $this->isMainAdmin = in_array($this->chatId, Config::ADMIN_IDS);
        
        // دریافت لیست ادمین‌های فرعی از دیتابیس
        $stmtAdmin = $this->db->prepare("SELECT value FROM settings WHERE `key` = 'bot_admins'");
        $stmtAdmin->execute();
        $dynAdminsStr = $stmtAdmin->fetchColumn();
        $dynAdminsArr = $dynAdminsStr ? explode(',', $dynAdminsStr) : [];
        
        // کاربر ادمین است اگر: ادمین اصلی باشد یا در لیست ادمین‌های فرعی دیتابیس باشد
        $this->isAdmin = $this->isMainAdmin || in_array($this->chatId, $dynAdminsArr);
        }

public function handle() {
        if (!$this->chatId) return;

        try {
            // 1. احراز هویت
            $user = $this->getUserOrRegister();

// ==============================================
            //            اولویت 0: بررسی قفل کانال
            // ==============================================
            if (!$this->isAdmin) {
                // اگر دکمه چک کردن جوین رو زده بود، اجازه بده بره به هندل کالبک
                if ($this->text !== 'check_join') {
                    $joinStatus = $this->checkMandatoryJoin();
                    if ($joinStatus !== true) {
                        // اگر کاربر عضو نباشد، کیبورد شیشه‌ای رو میسازه
                        // اینجا از تابع کیبوردی که توی Helpers ساختیم استفاده میشه
                        $this->sendMessage("⛔️ <b>کاربر گرامی!</b>\n\nبرای استفاده از امکانات ربات، ابتدا باید در کانال‌های زیر عضو شوید:", \App\Helpers\Keyboard::mandatoryJoinKeyboard($joinStatus));
                        return; // ربات متوقف میشه و پایین‌تر نمیره
                    }
                }
            }
            
            // ==============================================
            //           اولویت ۱: دکمه انصراف (همیشه فعال)
            // ==============================================
            if ($this->text == '❌ انصراف') {
                $this->updateUserStep('none');
                $this->sendMessage("عملیات لغو شد. به منوی اصلی بازگشتید.", Keyboard::mainMenu($this->isAdmin));
                return;
            }

            // ==============================================
            //           اولویت ۲: دریافت عکس رسید
            // ==============================================
            if ($user['step'] == 'awaiting_receipt') {
                // الف) اگر عکس دریافت شد
                if (isset($this->update['message']['photo'])) {
                    $this->handleReceiptUpload($user);
                    return;
                }
                
                // ب) اگر متنی غیر از انصراف آمد
                if ($this->text) {
                    $this->sendMessage("⚠️ لطفاً فقط **عکس رسید** را ارسال کنید.\nاگر منصرف شدید، دکمه «❌ انصراف» را بزنید.", Keyboard::cancel());
                    return;
                }
            }

// اولویت: دریافت کد تخفیف
            if ($user['step'] == 'awaiting_coupon_code') {
                if ($this->text == '❌ انصراف') {
                    $this->updateUserStep('none');
                    $this->sendMessage("منوی اصلی:", Keyboard::mainMenu($this->isAdmin));
                    return;
                }
                
                // پردازش کد تخفیف
                $this->checkAndApplyCoupon($user, $this->text);
                return;
            }
            
// اولویت: بررسی استعلام (اگر کاربر لینک فرستاده)
            if ($user['step'] == 'awaiting_inquiry' && $this->text) {
                $this->processInquiry($this->text);
                return;
            }
            
if ($user['step'] == 'awaiting_agency_request') {
                
                // 1. دریافت اطلاعات کاربر (با هندل کردن نام خانوادگی)
                $lastName = $this->update['message']['chat']['last_name'] ?? ''; 
                $fullName = htmlspecialchars($this->firstName . ' ' . $lastName);
                
                $userId = $this->chatId;
                $username = $this->username ? "@" . $this->username : "ندارد";
                $userMsg = htmlspecialchars($this->text);

                // 2. ساخت پیام برای ادمین‌ها
                $adminMsg = "<b>📩 درخواست نمایندگی جدید</b>\n\n";
                $adminMsg .= "👤 <b>کاربر:</b> <a href='tg://user?id=$userId'>$fullName</a>\n";
                $adminMsg .= "🆔 <b>آیدی:</b> $username\n";
                $adminMsg .= "🔢 <b>آیدی عددی:</b> <code>$userId</code>\n";
                $adminMsg .= "➖➖➖➖➖➖➖➖\n";
                $adminMsg .= "📝 <b>متن پیام:</b>\n\n";
                $adminMsg .= "$userMsg";

                // 3. ارسال برای تمام ادمین‌های لیست Config::ADMIN_IDS
                $sentCount = 0; // شمارش تعداد ارسال‌های موفق
                
                foreach (\App\Config\Config::ADMIN_IDS as $adminId) {
                    $req = $this->req('sendMessage', [
                        'chat_id' => $adminId,
                        'text' => $adminMsg,
                        'parse_mode' => 'HTML' 
                    ]);
                    
                    if (isset($req['ok']) && $req['ok']) {
                        $sentCount++;
                    }
                }

                // 4. نتیجه به کاربر
                // اگر حداقل برای یک ادمین هم رفته باشد، یعنی موفقیت‌آمیز بوده
                if ($sentCount > 0) {
                    $this->updateUserStep('none');
                    $this->sendMessage("✅ **پیام شما با موفقیت ثبت شد.**\nمدیریت در اسرع وقت پیام شما را بررسی کرده و در صورت تایید با شما تماس می‌گیرد.", Keyboard::mainMenu($this->isAdmin));
                } else {
                    $this->sendMessage("❌ خطا در ارسال پیام به مدیریت. لطفاً بعداً تلاش کنید.");
                }
                return; // ⛔️ حتماً باید باشد
            }            
            // ==============================================
            //           اولویت ۳: دستورات ادمین (تایید/رد)
            // ==============================================
            if ($this->isAdmin) {
                // کلیک روی دکمه تایید یا رد پرداخت
                if (isset($this->update['callback_query']) && strpos($this->text, 'admin_pay_') === 0) {
                    $this->handleAdminPaymentAction();
                    return;
                }
                // پنل مدیریت
                $this->handleAdminPanel($user);
            }

            // ==============================================
            //           اولویت ۴: دکمه‌های شیشه‌ای (خرید و...)
            // ==============================================
            if (isset($this->update['callback_query'])) {
                $this->handleCallbacks($user);
                return;
            }

            // ==============================================
            //           اولویت ۵: دستورات متنی منوی اصلی
            // ==============================================
            $this->handleTextCommands($user);

        } catch (Exception $e) {
            Logger::log("Error: " . $e->getMessage());
        }
    }
    
    // ==========================================
    //            مدیریت بخش‌های مختلف
    // ==========================================

// ==========================================
    //           مدیریت کلیک دکمه‌های شیشه‌ای
    // ==========================================
    private function handleCallbacks($user) {
        $data = $this->text; // دیتای دکمه شیشه‌ای

    // بررسی دکمه عضویت اجباری
        if ($data == 'check_join') {
            $joinStatus = $this->checkMandatoryJoin();
            if ($joinStatus === true) {
                // اگر عضو شده بود
                $this->deleteMessage($this->messageId);
                $this->sendMessage("✅ <b>عضویت شما تایید شد!</b>\nبه ربات خوش آمدید.", Keyboard::mainMenu($this->isAdmin));
            } else {
                // اگر هنوز عضو نشده
                $this->answerCallbackQuery("❌ شما هنوز در تمام کانال‌ها عضو نشده‌اید!", true);
            }
            return;
        }
        
        
        // ====================================================
        // 0. انتخاب نوع سرویس (srv_) -> نمایش لوکیشن‌ها (آلمان / انگلیس)
        // ====================================================
        if (strpos($data, 'srv_') === 0) {
            $this->editMessageText(
                "🌍 <b>انتخاب لوکیشن:</b>\n\nلطفاً کشور سرور را انتخاب کنید:", 
                Keyboard::locationsList()
            );
            $this->answerCallbackQuery("نوع سرویس انتخاب شد");
            return;
        }

        // ====================================================
        // 1. انتخاب لوکیشن (loc_) -> نمایش لیست زمان‌ها
        // ====================================================
        if (strpos($data, 'loc_') === 0) {
            $location = str_replace('loc_', '', $data);
            
            // ترجمه لوکیشن برای نمایش شیک‌تر
            $locNames = [
                'germany' => '🇩🇪 آلمان',
                'uk'      => '🇬🇧 انگلیس',
                'dedicated' => 'فول اختصاصی' // برای محکم کاری اگر دیتای قدیمی بود
            ];
            $displayLoc = $locNames[$location] ?? $location;

            $msg = "🌍 <b>لوکیشن $displayLoc</b>\n\n";
            $msg .= "⏳ لطفاً <b>مدت زمان</b> سرویس را انتخاب کنید:";

            $this->editMessageText(
                $msg,
                Keyboard::durationsList($location)
            );
            $this->answerCallbackQuery("لوکیشن انتخاب شد");
            return;
        }

        // ====================================================
        // 2. انتخاب زمان (dur_) -> نمایش لیست پلن‌ها (حجم‌ها)
        // ====================================================
        if (strpos($data, 'dur_') === 0) {
            // فرمت دیتا: dur_لوکیشن|زمان
            $content = str_replace('dur_', '', $data);
            $parts = explode('|', $content);
            
            if (count($parts) >= 2) {
                $location = $parts[0];
                $duration = $parts[1];
                
                $locNames = ['germany' => '🇩🇪 آلمان', 'uk' => '🇬🇧 انگلیس', 'dedicated' => '💎 فول اختصاصی'];
                $displayLoc = $locNames[$location] ?? $location;

                $msg = "<b>سرویس‌های $duration - $displayLoc</b>\n\n";
                $msg .= "📦 یکی از پلن‌های زیر را انتخاب کنید:";

                $this->editMessageText(
                    $msg, 
                    Keyboard::plansList($location, $duration)
                );
            }
            $this->answerCallbackQuery("زمان انتخاب شد");
            return;
        }

        // ====================================================
        // 3. انتخاب نهایی پلن (buy_plan_) -> نمایش فاکتور
        // ====================================================
        if (strpos($data, 'buy_plan_') === 0) {
            $planId = str_replace('buy_plan_', '', $data);
            
            // ذخیره آیدی پلن در تمپ تا اگر کد تخفیف زد یادمان بماند
            $this->updateUserStep('none', $planId);
            
            // نمایش فاکتور خرید
            $this->showInvoice($planId); 
            
            $this->answerCallbackQuery("🧾 فاکتور صادر شد");
            return;
        }

        // ====================================================
        // 4. دکمه‌های بازگشت (Navigation)
        // ====================================================
        
    // بازگشت به مرحله انتخاب زمان (حذف لوکیشن)
        if ($data == 'back_to_main' || $data == 'back_to_service_type') {
            $msg = "💎 <b>خرید سرویس فول اختصاصی</b>\n\n";
            $msg .= "⏳ لطفاً <b>مدت زمان</b> سرویس مورد نظر خود را انتخاب کنید:";
            $this->editMessageText($msg, Keyboard::durationsList('dedicated'));
            return;
        }
        // بازگشت به منوی اصلی ربات (از همه جا)
        if ($data == 'back_to_main_menu') {
            $this->deleteMessage($this->messageId);
            $this->sendMessage("منوی اصلی:", Keyboard::mainMenu($this->isAdmin));
            return;
        }

        // ====================================================
        // 5. مدیریت کد تخفیف در مرحله فاکتور
        // ====================================================
        if (strpos($data, 'enter_coupon_') === 0) {
            $planId = str_replace('enter_coupon_', '', $data);
            // ذخیره آیدی پلن در تمپ برای مرحله بعد
            $this->updateUserStep('awaiting_coupon_code', $planId);
            $this->deleteMessage($this->messageId); // حذف فاکتور قبلی
            $this->sendMessage("🎟 <b>لطفاً کد تخفیف خود را ارسال کنید:</b>", Keyboard::cancel());
            return;
        }

        if (strpos($data, 'remove_coupon_') === 0) {
            $planId = str_replace('remove_coupon_', '', $data);
            $this->updateUserStep('none'); // پاک کردن کد از استپ
            $this->showInvoice($planId, null); // نمایش فاکتور بدون کد
            $this->answerCallbackQuery("کد تخفیف حذف شد.");
            return;
        }

        // ====================================================
        // 6. پرداخت‌ها (کارت به کارت / کیف پول)
        // ====================================================

        // پرداخت کارت به کارت
        if (strpos($data, 'pay_card_') === 0) {
            $planId = str_replace('pay_card_', '', $data);
            
            $this->updateUserStep('awaiting_receipt', $planId);
            
            $number = $this->getSetting('card_number');
            $name = $this->getSetting('card_name');
            
            if (!$number) {
                $this->answerCallbackQuery("⚠️ شماره کارت در سیستم ثبت نشده است.", true);
                return;
            }
            
            $safeName = $name ? htmlspecialchars($name) : "نامشخص";
            
            $msg = "💳 <b>پرداخت کارت به کارت</b>\n\n";
            $msg .= "لطفاً مبلغ را به شماره کارت زیر واریز کنید:\n";
            $msg .= "👇 <b>(برای کپی روی شماره کلیک کنید)</b>\n\n";
            $msg .= "<code>$number</code>\n";
            $msg .= "👤 <b>$safeName</b>\n\n";
            $msg .= "📸 <b>سپس عکس رسید را همینجا ارسال کنید.</b>";
            
            $this->deleteMessage($this->messageId);
            $this->sendMessage($msg, Keyboard::cancel());
            return;
        }

        // پرداخت با کیف پول
        if (strpos($data, 'pay_wallet_') === 0) {
            $planId = str_replace('pay_wallet_', '', $data);
            $this->processWalletPayment($user, $planId);
            return;
        }

        // ====================================================
        // 7. سایر دکمه‌ها (تست رایگان، سرویس من و...)
        // ====================================================

        // دریافت تست رایگان
        if ($data == 'get_free_test') {
            $this->processFreeTest($user);
            return;
        }

        // مشاهده جزئیات سرویس خریداری شده
        if (strpos($data, 'my_service_') === 0) {
            $orderId = str_replace('my_service_', '', $data);
            $this->showServiceDetails($orderId);
            return;
        }
    }
    
    private function handleTextCommands($user) {
        if (strpos($this->text, '/start') === 0) {
            $this->text = '/start';
        }
        
        switch ($this->text) {
case '/start':
            case '🔙 بازگشت':
            case '❌ انصراف':
                $this->updateUserStep('none');
                
                // دریافت متن از دیتابیس
                $msg = $this->getSetting('start_text');
                
                // اگر متنی تنظیم نکرده باشید، این پیام موقت را نشان می‌دهد
                if (!$msg) {
                    $msg = "به ربات ما خوش آمدید! 👋\n\nجهت تغییر این متن، از پنل مدیریت گزینه «📝 ویرایش متن استارت» را انتخاب کنید.";
                }

                $this->sendMessage($msg, Keyboard::mainMenu($this->isAdmin));
                break;
                case '👱 سرویس های من':
                // اول باید ID کاربر در دیتابیس را پیدا کنیم
                $stmt = $this->db->prepare("SELECT id FROM users WHERE chat_id = ?");
                $stmt->execute([$this->chatId]);
                $dbId = $stmt->fetchColumn();
                
                $this->sendMessage("📂 **لیست سرویس‌های شما:**\n\nجهت مشاهده جزئیات و لینک اتصال، روی سرویس مورد نظر کلیک کنید:", Keyboard::userServicesList($dbId));
                break;
case '📚 آموزش ها':
                $msg .= "جهت مشاهده آموزش‌های اتصال، لطفاً به کانال آموزشی ما مراجعه کنید:\n\n";
                $msg .= "🆔 @msvpn_sting";
                
                $this->sendMessage($msg, Keyboard::mainMenu($this->isAdmin));
                break;
                
            case '🔎 استعلام':
                $this->updateUserStep('awaiting_inquiry');
                $this->sendMessage("🔎 **بخش استعلام سرویس**\n\nلطفاً **لینک کانفیگ** (vmess/vless) یا **کد UUID** خود را ارسال کنید تا وضعیت آن را بررسی کنم:", Keyboard::cancel());
                break;
case '🛍 خرید سرویس':
                $msg = "💎 <b>خرید سرویس فول اختصاصی</b>\n\n";
                $msg .= "⏳ لطفاً <b>مدت زمان</b> سرویس مورد نظر خود را انتخاب کنید:";
                $this->sendMessage($msg, Keyboard::durationsList('dedicated'));
                break;
                case '🔥 تست رایگان':
                $desc = $this->getSetting('test_description') ?: "جهت دریافت تست رایگان دکمه زیر را بزنید:";
                $this->sendMessage($desc, Keyboard::testInline());
                break;

case '🤝 زیرمجموعه گیری':
                // تعداد دعوت‌های موفق این کاربر
                $stmt = $this->db->prepare("SELECT COUNT(*) FROM users WHERE invited_by = ?");
                $stmt->execute([$this->chatId]);
                $inviteCount = $stmt->fetchColumn();

                // 🔴 بسیار مهم: آیدی ربات خودتان را در خط زیر به جای YOUR_BOT_USERNAME قرار دهید (بدون @)
                $botUsername = "msvpnorg1_bot"; 
                $refLink = "https://t.me/$botUsername?start={$this->chatId}";

                $msg = "🤝 <b>سیستم زیرمجموعه گیری و کسب درآمد</b>\n\n";
                $msg .= "با دعوت از دوستان خود به ربات، <b>5%</b> از مبلغ هر خرید آنها را به صورت خودکار به عنوان پورسانت در کیف پول خود دریافت کنید!\n\n";                $msg .= "👥 تعداد زیرمجموعه‌های شما: <b>$inviteCount نفر</b>\n\n";
                $msg .= "👇 <b>لینک اختصاصی دعوت شما:</b>\n";
                $msg .= "<code>$refLink</code>\n\n";
                $msg .= "این لینک را کپی کرده و برای دوستان خود ارسال کنید.";
                
                $this->sendMessage($msg, Keyboard::mainMenu($this->isAdmin));
                break;
                
            case '🆔 آیدی من':
                $bal = number_format($user['wallet']);
                $msg = "👤 <b>پروفایل کاربری:</b>\n\n";
                $msg .= "🔸 نام: <b>$this->firstName</b>\n";
                $msg .= "🔸 شناسه عددی: <code>$this->chatId</code>\n";
                $msg .= "💰 موجودی کیف پول: <b>$bal تومان</b>";
                $this->sendMessage($msg, Keyboard::backOnly());
                break;
case '🤝 درخواست نمایندگی':
                $this->updateUserStep('awaiting_agency_request');
                $this->sendMessage("📄 **فرم درخواست نمایندگی**\n\nلطفاً پیام، رزومه یا شرایط پیشنهادی خود را در قالب یک پیام متنی بنویسید و ارسال کنید.\n\nادمین پیام شما را بررسی کرده و با شما تماس خواهد گرفت.", Keyboard::cancel());
                break;
case '👨‍💻 پشتیبانی':
                // دریافت آیدی از دیتابیس
                $stmt = $this->db->prepare("SELECT value FROM settings WHERE `key` = 'support_id'");
                $stmt->execute();
                $supId = $stmt->fetchColumn();
                
                // اگر تنظیم نشده بود، پیش‌فرض بگذار
                if (!$supId) $supId = "@Admin"; 
                
                $msg = "📞 **پشتیبانی آنلاین**\n\n";
                $msg .= "برای ارتباط با پشتیبانی، روی آیدی زیر کلیک کنید:\n\n";
                $msg .= "🆔 <b>$supId</b>\n\n";
                $msg .= "ساعات پاسخگویی: ۱۰ صبح تا ۱۲ شب";
                
                $this->sendMessage($msg, Keyboard::backOnly());
                break;            
            case '💰 کیف پول':
                $bal = number_format($user['wallet']);
                $this->sendMessage("💰 <b>کیف پول شما:</b>\n\nموجودی فعلی: <b>$bal تومان</b>\n\nجهت شارژ حساب با پشتیبانی تماس بگیرید.", Keyboard::backOnly());
                break;

            default:
                // اگر ادمین نبود و دستور نامعتبر بود
                if (!$this->isAdmin) {
                    $this->sendMessage("⚠️ دستور نامعتبر است. لطفاً از دکمه‌های منو استفاده کنید.", Keyboard::mainMenu($this->isAdmin));
                }
        }
    }

    private function handleAdminPanel($user) {
        
        $userSideSteps = [
            'awaiting_receipt',
            'awaiting_inquiry',
            'awaiting_agency_request',
            'awaiting_coupon_code',
            'coupon_applied'
        ];

        // اگر مرحله جاری جزو مراحل کاربر بود، بیخیال پنل ادمین شو
        if (in_array($user['step'], $userSideSteps)) {
            return;
        }

        // 2. همچنین اگر دکمه‌ای که زده شد مربوط به نویگیشن کاربر بود (مثل انتخاب لوکیشن یا زمان)
        // باز هم بیخیال پنل ادمین شو (چون ادمین کنترلر این‌ها را نمی‌فهمد)
        if (isset($this->update['callback_query'])) {
            $cbData = $this->text;
            if (strpos($cbData, 'dur_') === 0 || 
                strpos($cbData, 'loc_') === 0 || 
                strpos($cbData, 'buy_plan_') === 0 || 
                strpos($cbData, 'enter_coupon_') === 0 || 
                strpos($cbData, 'remove_coupon_') === 0 || 
                $cbData == 'back_to_main' || 
                $cbData == 'back_to_main_menu') {
                return;
            }
        }
        
// 5. مدیریت پنل ادمین (لیست کامل دستورات)
            $adminCommands = [
                '⚙️ پنل مدیریت', 
                '👨‍✈️ مدیریت ادمین‌ها', 
                '➕ افزودن ادمین', 
                '🗑 حذف ادمین',
                '📝 ویرایش متن استارت',
                '➕ افزودن محصول جدید', 
                '❌ حذف محصول',
                '✏️ ویرایش متن تست', 
                '📢 ارسال همگانی', 
                '🔖 تخفیف نمایندگی', 
                '❌ انصراف', 
                '🆔 ویرایش آیدی پشتیبانی', // ✅ این جدید است
                '📊 آمار ربات', 
                '💳 ویرایش شماره کارت',
                '💰 مدیریت موجودی', // ✅ این خط جا افتاده بود که اضافه شد
                // --- دکمه‌های جدید مدیریت لوکیشن و زمان ---
                '🌍 مدیریت لوکیشن‌ها',
                '⏳ مدیریت زمان‌ها',
                '➕ افزودن لوکیشن',
                '🗑 حذف لوکیشن',
                '🎟 کد تخفیف',          // ✅ جدید
            '➕ ساخت کد تخفیف',      // ✅ جدید
            '⚙️ مدیریت کد تخفیف ها', // ✅ جدید
                '➕ افزودن زمان',
                '🗑 حذف زمان',
                '🔢 تغییر شماره کارت',
                '📢 قفل کانال',
                '🗑 حذف کانال',
                '➕ افزودن کانال جدید',
                '👤 تغییر نام صاحب حساب',
                '🔙 بازگشت به پنل مدیریت',
                                '🔙 بازگشت به پنل کاربری',
                '🔙 بازگشت' // این هم باشد برای زیرمنوها
            ];

            // شرط بررسی: اگر ادمین است و (پیام جزو دستورات بالاست یا کاربر در مرحله‌ای از ادمین است)
// شرط بررسی: اگر ادمین است و (پیام جزو دستورات بالاست یا کاربر در مرحله‌ای از ادمین است)
            if ($this->isAdmin && ($user['step'] != 'none' || in_array($this->text, $adminCommands))) {
                if (class_exists('App\Controllers\AdminController')) {
                    $adminCtrl = new AdminController($this->chatId, $this->messageId, $this->isMainAdmin);
                    list($msg, $kb) = $adminCtrl->handleRequest($this->text, $user['step'], $user['temp_data']);
                    
                    $this->sendMessage($msg, $kb);
                    exit; // توقف اجرا تا دستورات پایین اجرا نشوند
                }
            }
            }

    // ==========================================
    //            توابع منطقی (Logic)
    // ==========================================

    private function showInvoiceById($planId) {
        $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ? AND is_active = 1");
        $stmt->execute([$planId]);
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($plan) {
            $price = number_format($plan['price']);
            $gb = $plan['volume_gb'];
            $days = $plan['days'];
            $loc = $plan['location'] ?? '🌍';

            // طراحی فاکتور شیک و تمیز
            $msg = "🧾 <b>پیش‌فاکتور خرید</b>\n\n";
            $msg .= "💎 <b>سرویس:</b> $loc\n";
            $msg .= "📊 <b>حجم:</b> $gb گیگابایت\n";
            $msg .= "⏳ <b>مدت زمان:</b> $days روز\n";
            $msg .= "➖➖➖➖➖➖➖➖\n";
            $msg .= "💰 <b>مبلغ قابل پرداخت:</b> $price تومان\n\n";
            $msg .= "👇 لطفاً روش پرداخت را انتخاب کنید:";

            // به جای ارسال پیام جدید، پیام قبلی (لیست پلن‌ها) را ادیت میکنیم
            $this->editMessageText($msg, Keyboard::invoiceButtons($plan['id'], $plan['price']));
        } else {
            $this->answerCallbackQuery("⚠️ این محصول دیگر موجود نیست!", true);
            $this->editMessageText("⛔️ متاسفانه این محصول ناموجود یا غیرفعال شده است.", Keyboard::locationsList());
        }
    }


// ==========================================
    // پردازش خرید با کیف پول (اصلاح شده نهایی)
    // ==========================================
    private function processWalletPayment($user, $pid) {
        $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ?");
        $stmt->execute([$pid]);
        $plan = $stmt->fetch(\PDO::FETCH_ASSOC);
        
        if (!$plan) {
            $this->sendMessage("❌ محصول یافت نشد.");
            return;
        }

        // --- محاسبه قیمت ---
        $resellerPercent = $user['discount_percent'] ?? 0;
        $priceStep1 = $plan['price'];
        
        if ($resellerPercent > 0) {
            $priceStep1 = $plan['price'] - (($plan['price'] * $resellerPercent) / 100);
        }

        $couponCode = null;
        $finalPrice = $priceStep1;
        
        if ($user['step'] == 'coupon_applied' && strpos($user['temp_data'], '|') !== false) {
            $parts = explode('|', $user['temp_data']);
            if ($parts[0] == $pid) {
                $couponCode = $parts[1];
                $stmtC = $this->db->prepare("SELECT * FROM coupons WHERE code = ?");
                $stmtC->execute([$couponCode]);
                $couponData = $stmtC->fetch(\PDO::FETCH_ASSOC);
                
                if ($couponData && $couponData['used_count'] < $couponData['max_uses']) {
                    $discountAmt = ($priceStep1 * $couponData['percent']) / 100;
                    $finalPrice = $priceStep1 - $discountAmt;
                } else {
                    $couponCode = null;
                }
            }
        }

        // --- عملیات خرید ---
        if ($user['wallet'] >= $finalPrice) {
            
            $this->db->prepare("UPDATE users SET wallet = wallet - ? WHERE chat_id = ?")->execute([$finalPrice, $this->chatId]);
            
            $this->answerCallbackQuery("⏳ در حال صدور آنی...");
            $this->sendMessage("⏳ **در حال ساخت سرویس...**\nلطفاً چند لحظه صبر کنید.");

            $panel = new \App\Helpers\V2RayPanel();
            $username = "User_" . $this->chatId . "_" . rand(100, 999);
            
            // 🔴 اصلاح مهم: پارامتر آخر (UUID) را null می‌فرستیم
$targetInbound = $plan['inbound_id'] ?? 2; 
            $result = $panel->addUser($targetInbound, $username, $plan['volume_gb'], $plan['days'], null);
            if ($result && isset($result['success']) && $result['success']) {
                
                $uuid = $result['uuid'];

$configLink = $this->generateSubLink($uuid, $username, $plan['location']);                
                $this->db->prepare("INSERT INTO orders (user_id, plan_id, uuid, total_price, status, created_at) VALUES (?, ?, ?, ?, 'completed', NOW())")
                         ->execute([$user['id'], $pid, $uuid, $finalPrice]);
$this->giveReferralReward($this->chatId, $finalPrice);

                if ($couponCode) {
                    $this->db->prepare("UPDATE coupons SET used_count = used_count + 1 WHERE code = ?")->execute([$couponCode]);
                }

                $msg = "✅ **خرید با موفقیت انجام شد!**\n\n";
                $msg .= "📦 <b>سرویس:</b> {$plan['name']}\n";
                $msg .= "👤 <b>نام کاربری:</b> <code>$username</code>\n";
                $msg .= "💰 <b>مبلغ کسر شده:</b> " . number_format($finalPrice) . " تومان\n\n";
                $msg .= "🔌 <b>لینک اتصال شما:</b>\n\n";
                $msg .= "<pre>$configLink</pre>\n\n";
                
                $this->sendMessage($msg, Keyboard::mainMenu($this->isAdmin));
                $this->updateUserStep('none');

            } else {
                $this->refundUser($finalPrice, "خطا در اتصال به پنل (موجودی برگشت خورد)");
            }
        } else {
            $this->answerCallbackQuery("⛔️ موجودی کافی نیست!", true);
            $this->sendMessage("⛔️ <b>موجودی کیف پول شما کافی نیست.</b>\n\n💰 هزینه سرویس: " . number_format($finalPrice) . " تومان\n💳 موجودی شما: " . number_format($user['wallet']) . " تومان\n\nلطفاً حساب خود را شارژ کنید.");
        }
    }
    // تابع کمکی برای برگشت پول (Refund)
    private function refundUser($amount, $reason) {
        $this->db->prepare("UPDATE users SET wallet = wallet + ? WHERE chat_id = ?")->execute([$amount, $this->chatId]);
        $this->sendMessage("❌ <b>خرید ناموفق بود.</b>\n\nدلیل: $reason\n💰 مبلغ <b>" . number_format($amount) . " تومان</b> به کیف پول شما برگشت داده شد.");
    }    
    // پردازش آپلود رسید
// پردازش آپلود رسید توسط کاربر
private function handleReceiptUpload($user) {
        $planId = $user['temp_data']; // آیدی محصول که قبلا ذخیره کردیم
        
        // دریافت اطلاعات محصول
        $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ?");
        $stmt->execute([$planId]);
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$plan) {
            $this->updateUserStep('none');
            $this->sendMessage("⚠️ محصول مورد نظر یافت نشد. لطفاً مجدد تلاش کنید.");
            return;
        }

        // دریافت بزرگترین سایز عکس
        $photoArray = $this->update['message']['photo'];
        $photoId = end($photoArray)['file_id'];
        
        // --- 👈 تبدیل لوکیشن انگلیسی به فارسی با پرچم 👉 ---
        $locNames = [
            'germany' => '🇩🇪 آلمان',
            'uk'      => '🇬🇧 انگلیس',
            'dedicated' => '💎 فول اختصاصی'
        ];
        $displayLoc = isset($locNames[$plan['location']]) ? $locNames[$plan['location']] : $plan['location'];

        // ساخت کپشن برای ادمین
        $caption = "🧾 **رسید جدید دریافت شد!**\n\n";
        $caption .= "👤 کاربر: <a href='tg://user?id=$this->chatId'>$this->firstName</a>\n";
        $caption .= "🆔 آیدی عددی: <code>$this->chatId</code>\n";
        $caption .= "📦 سرویس: {$plan['name']}\n";
        $caption .= "💰 قیمت: " . number_format($plan['price']) . " تومان\n";
        $caption .= "🌍 لوکیشن: $displayLoc\n"; // 👈 مشکل اینجا بود که حل شد
        $caption .= "⏳ مدت: {$plan['days']} روز\n\n";
        $caption .= "👇 **عملیات:**";

        // ساخت دکمه‌های تایید و رد (به همراه آیدی کاربر و آیدی پلن)
        $kb = json_encode(['inline_keyboard' => [[
            ['text' => '✅ تایید و تحویل', 'callback_data' => "admin_pay_accept_{$this->chatId}_{$planId}"],
            ['text' => '❌ رد کردن', 'callback_data' => "admin_pay_reject_{$this->chatId}"]
        ]]]);

        // ارسال عکس برای تمام ادمین‌ها
        foreach (\App\Config\Config::ADMIN_IDS as $adminId) {
            $this->sendPhoto($adminId, $photoId, $caption, $kb);
        }
        
        // تغییر وضعیت کاربر و تشکر
        $this->updateUserStep('none');
        $this->sendMessage("✅ **رسید شما دریافت شد.**\n\nادمین‌ها در حال بررسی هستند. پس از تایید، سرویس شما همینجا ارسال می‌شود.\nلطفاً منتظر باشید.", Keyboard::mainMenu($this->isAdmin));
    }
    // عملیات تایید یا رد پرداخت توسط ادمین

// ==========================================
    // مدیریت تایید یا رد پرداخت توسط ادمین (اصلاح شده نهایی)
    // ==========================================
    private function handleAdminPaymentAction() {
        $data = $this->text; 

        // 🟢 حالت اول: تایید پرداخت (Accept)
        if (strpos($data, 'admin_pay_accept_') === 0) {
            
            $parts = explode('_', str_replace('admin_pay_accept_', '', $data));
            
            if (count($parts) < 2) {
                $this->answerCallbackQuery("❌ فرمت داده نامعتبر است.", true);
                return;
            }

            $targetChatId = $parts[0];
            $planId = $parts[1];

            $this->answerCallbackQuery("⏳ در حال ساخت سرویس...");
            $this->editMessageCaption($this->chatId, $this->messageId, "⏳ **در حال اتصال به سرور...**");

            // 1. دریافت اطلاعات پلن
            $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ?");
            $stmt->execute([$planId]);
            $plan = $stmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$plan) {
                $this->answerCallbackQuery("❌ خطا: محصول یافت نشد.", true);
                return;
            }

            // 2. اتصال به پنل
            $panel = new \App\Helpers\V2RayPanel();
            
            // نام کاربری رندوم
            $username = "User_" . $targetChatId . "_" . rand(100, 999);
            
            // 🔴 اصلاح مهم: پارامتر آخر (UUID) را null می‌فرستیم تا خود تابع یکی بسازد
            // ورودی‌ها: (InboundID, Username, Volume, Days, UUID)
// خواندن آیدی پنل (اینباند) مستقیماً از محصولی که کاربر خریده است
            $targetInbound = $plan['inbound_id'] ?? 2;
            $createResult = $panel->addUser($targetInbound, $username, $plan['volume_gb'], $plan['days'], null);
            if ($createResult && isset($createResult['success']) && $createResult['success']) {
                
                // دریافت UUID ساخته شده
                $uuid = $createResult['uuid'];

                // ساخت لینک
$configLink = $this->generateSubLink($uuid, $username, $plan['location']);                
                // 3. ثبت سفارش
                $stmtU = $this->db->prepare("SELECT id FROM users WHERE chat_id = ?");
                $stmtU->execute([$targetChatId]);
                $dbUserId = $stmtU->fetchColumn();

                if ($dbUserId) {
                    $this->db->prepare("INSERT INTO orders (user_id, plan_id, uuid, total_price, status, created_at) VALUES (?, ?, ?, ?, 'completed', NOW())")
                             ->execute([$dbUserId, $planId, $uuid, $plan['price']]);
                             $this->giveReferralReward($targetChatId, $plan['price']);
                }

                // 4. پیام به کاربر
                $msgUser = "✅ **سفارش شما تایید و فعال شد!**\n\n";
                $msgUser .= "📦 <b>سرویس:</b> {$plan['name']}\n";
                $msgUser .= "👤 <b>نام کاربری:</b> <code>$username</code>\n";
                $msgUser .= "⏳ <b>اعتبار:</b> {$plan['days']} روز\n";
                $msgUser .= "📊 <b>حجم:</b> {$plan['volume_gb']} گیگابایت\n\n";
                $msgUser .= "🔌 <b>لینک اتصال اختصاصی شما:</b>\n\n";
                $msgUser .= "<pre>$configLink</pre>\n\n";
                $msgUser .= "❤️ از خرید شما متشکریم.";
                
                $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=400x400&margin=15&data=" . urlencode($configLink);
                
                // 👈 ارسال عکس به همراه متن به عنوان کپشن
                $this->req('sendPhoto', [
                    'chat_id' => $targetChatId,
                    'photo' => $qrUrl,
                    'caption' => $msgUser,
                    'parse_mode' => 'HTML'
                ]);
                

                // 5. پیام به ادمین
                $adminMsg = "✅ **تایید و تحویل داده شد.**\n\n";
                $adminMsg .= "👤 خریدار: <a href='tg://user?id=$targetChatId'>$targetChatId</a>\n";
                $adminMsg .= "📦 سرویس: {$plan['name']}\n";
                $adminMsg .= "🔑 یوزرنیم: <code>$username</code>\n"; // نمایش یوزرنیم ساخته شده
                $adminMsg .= "💰 مبلغ: " . number_format($plan['price']);
                
                $this->editMessageCaption($this->chatId, $this->messageId, $adminMsg);
                $this->answerCallbackQuery("انجام شد ✅");

            } else {
                $this->editMessageCaption($this->chatId, $this->messageId, "❌ **خطا:** اکانت در پنل ساخته نشد. لاگ را چک کنید.");
            }
        } 
        
        // 🔴 حالت رد پرداخت
        elseif (strpos($data, 'admin_pay_reject_') === 0) {
            $targetChatId = str_replace('admin_pay_reject_', '', $data);
            $this->sendMessageToId($targetChatId, "❌ **متاسفانه پرداخت شما تایید نشد.**\n\nجهت پیگیری با پشتیبانی تماس بگیرید.");
            $this->editMessageCaption($this->chatId, $this->messageId, "❌ **توسط شما رد شد.**");
            $this->answerCallbackQuery("رد شد ❌");
        }
    }
    
// ==========================================
    // پردازش تست رایگان (نسخه هیدیفای)
    // ==========================================
    private function processFreeTest($user) {
        // 1. بررسی اینکه کاربر قبلاً تست گرفته یا خیر (جلوگیری از سوءاستفاده)
        if ($user['got_free_test'] == 1 && !$this->isAdmin) {
            $this->answerCallbackQuery("⛔️ شما قبلاً سرویس تست رایگان را دریافت کرده‌اید.", true);
            return;
        }

        // نمایش پیام انتظار
        $this->answerCallbackQuery("🚀 در حال آماده‌سازی...", false);
        $this->sendMessage("⏳ <b>در حال ساخت سرویس تست...</b>\nلطفاً چند لحظه صبر کنید.");

        $panel = new \App\Helpers\V2RayPanel();
        
        // تولید نام کاربری رندوم برای تست
        $username = "Test_" . $this->chatId . "_" . rand(100, 999);
        
        // تنظیمات هیدیفای: 0.2 گیگابایت (200 مگ) - 1 روز
        // پارامتر اول (اینباند) در هیدیفای بی‌تاثیر است، پس عدد 1 می‌فرستیم
        $result = $panel->addUser(1, $username, 0.2, 1, null);
        
        if ($result && isset($result['success']) && $result['success']) {

            $uuid = $result['uuid'];

            // تولید لینک اختصاصی (سابسکرایب هیدیفای)
            $finalConfigText = $this->generateSubLink($uuid, $username);

            // ذخیره در دیتابیس که کاربر تست گرفته است
            $this->db->prepare("UPDATE users SET got_free_test = 1, test_uuid = ?, test_created_at = NOW() WHERE chat_id = ?")
                     ->execute([$uuid, $this->chatId]);
            
            // --- طراحی پیام جذاب ---
            $msg = "🎁 <b>سرویس تست رایگان شما فعال شد</b>\n\n";
            $msg .= "👤 <b>نام کاربری:</b> <code>$username</code>\n";
            $msg .= "⏳ <b>اعتبار سرویس:</b> ۱ روز\n";
            $msg .= "📊 <b>حجم اختصاصی:</b> ۲۰۰ مگابایت\n\n";
            
            $msg .= "🔌 <b>لینک اتصال اختصاصی شما:</b>\n\n";
            $msg .= "<code>$finalConfigText</code>\n\n";
            $msg .= "🚀 <b>نهایت سرعت و امنیت</b>";
            
            // تولید لینک عکس QR Code
            $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=400x400&margin=15&data=" . urlencode($finalConfigText);
            
            // تلاش برای پاک کردن پیام "در حال ساخت..."
            $this->deleteMessage($this->messageId + 1); 
            
            // ارسال پیام نهایی به همراه عکس بارکد
            $this->req('sendPhoto', [
                'chat_id' => $this->chatId,
                'photo' => $qrUrl,
                'caption' => $msg,
                'parse_mode' => 'HTML',
                'reply_markup' => Keyboard::mainMenu($this->isAdmin)
            ]);
            
        } else {
            $this->sendMessage("❌ خطا در ارتباط با سرور هیدیفای. لطفاً دقایقی دیگر تلاش کنید.");
        }
    }    
    // تابع قدرتمند دانلود سابسکرایب
    private function fetchSubscription($url) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_USERAGENT, 'v2rayNG/1.8.5');
        // هدرهای اضافی برای جلوگیری از بلاک شدن
        curl_setopt($ch, CURLOPT_ENCODING, ""); // Handle gzip
        
        $response = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code == 200 && !empty($response)) {
            // دیکود بیس ۶۴
            $decoded = base64_decode($response, true);
            if ($decoded && (strpos($decoded, 'vless://') !== false || strpos($decoded, 'vmess://') !== false)) {
                return trim($decoded);
            }
            // اگر متن ساده بود
            if (strpos($response, 'vless://') !== false) {
                return trim($response);
            }
        }
        return null;
    }    
    // --- تابع قدرتمند برای دانلود و دیکود سابسکرایب ---
    private function getDecodedSub($url) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_USERAGENT, 'v2rayNG/1.8.5'); 
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        
        $response = curl_exec($ch);
        $err = curl_error($ch);
        curl_close($ch);

        if ($err || empty($response)) {
            // اگر نتوانست دانلود کند، نال برگردان
            return null;
        }

        // 1. تلاش برای دیکود کردن Base64 (مرزبان همیشه Base64 می‌دهد)
        $decoded = base64_decode($response, true);
        
        // 2. بررسی اعتبار دیکود
        if ($decoded) {
            // اگر دیکود شد، معمولاً لیستی از vless:// و vmess:// است که با اینتر جدا شده‌اند
            return trim($decoded);
        }

        // 3. اگر دیکود نشد اما متن ساده بود (شاید ساب شما انکریپت ندارد)
        if (strpos($response, 'vless://') !== false || strpos($response, 'vmess://') !== false) {
            return trim($response);
        }

        return null;
    }
    
    // ==========================================
    //            توابع کمکی و زیرساختی
    // ==========================================

private function getUserOrRegister() {
        $chatId = $this->chatId;
        $firstName = $this->firstName;
        $username = $this->username;

        // ۱. ابتدا چک می‌کنیم کاربر از قبل در دیتابیس هست یا نه
        $stmt = $this->db->prepare("SELECT * FROM users WHERE chat_id = ?");
        $stmt->execute([$chatId]);
        $existingUser = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$existingUser) {
            // ۲. کاربر جدید است! بررسی لینک دعوت
            $invitedBy = null;
            if ($this->text && strpos($this->text, '/start ') === 0) {
                $refId = str_replace('/start ', '', $this->text);
                if (is_numeric($refId) && $refId != $chatId) {
                    $invitedBy = $refId;
                    
                    // 👈 ارسال پیام اطلاع‌رسانی به فرد دعوت‌کننده
                    $safeName = htmlspecialchars($firstName);
                    $this->sendMessageToId($invitedBy, "🎊 <b>کاربر جدید!</b>\n\nیک نفر ($safeName) با لینک اختصاصی شما وارد ربات شد.\nدر صورت خرید، 10% از مبلغ پرداختی او به کیف پول شما اضافه می‌شود. 💰");
                }
            }

            // ۳. ثبت کاربر جدید در دیتابیس
            $sql = "INSERT INTO users (chat_id, full_name, username, wallet, invited_by) 
                    VALUES (:chat_id, :full_name, :username, 0, :invited_by)";
            $stmtInsert = $this->db->prepare($sql);
            $stmtInsert->execute([
                ':chat_id' => $chatId,
                ':full_name' => $firstName,
                ':username' => $username,
                ':invited_by' => $invitedBy
            ]);
            
            // دریافت مجدد اطلاعات برای ادامه کار
            $stmt->execute([$chatId]);
            return $stmt->fetch(\PDO::FETCH_ASSOC);
            
        } else {
            // ۴. کاربر قدیمی است، فقط آپدیت نام و یوزرنیم
            $sql = "UPDATE users SET full_name = :full_name, username = :username WHERE chat_id = :chat_id";
            $this->db->prepare($sql)->execute([
                ':full_name' => $firstName,
                ':username' => $username,
                ':chat_id' => $chatId
            ]);
            
            return $existingUser;
        }
    }
    private function updateUserStep($step, $tempData = null) {
        $sql = "UPDATE users SET step = :step";
        if ($tempData !== null) $sql .= ", temp_data = :data";
        else $sql .= ", temp_data = NULL";
        
        $sql .= " WHERE chat_id = :id";
        
        $stmt = $this->db->prepare($sql);
        $params = [':step' => $step, ':id' => $this->chatId];
        if ($tempData !== null) $params[':data'] = $tempData;
        
        $stmt->execute($params);
    }

    private function getSetting($key) {
        $stmt = $this->db->prepare("SELECT value FROM settings WHERE `key` = ?");
        $stmt->execute([$key]);
        return $stmt->fetchColumn();
    }

// تابع تولید UUID پیشرفته و کاملاً یکتا (Cryptographically Secure)
    private function generateUUID() {
        try {
            // استفاده از random_bytes که غیرقابل پیش‌بینی است
            $data = random_bytes(16);
            $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // تنظیم نسخه ۴
            $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // تنظیم variant
            
            return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
        } catch (Exception $e) {
            // اگر سرور قدیمی بود و random_bytes نداشت، روش جایگزین:
            return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
                mt_rand(0, 0xffff), mt_rand(0, 0xffff),
                mt_rand(0, 0xffff),
                mt_rand(0, 0x0fff) | 0x4000,
                mt_rand(0, 0x3fff) | 0x8000,
                mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
            );
        }
    }
    // --- توابع ارتباط با تلگرام (cURL) ---
    
private function sendMessage($text, $kb = null) { 
        return $this->req('sendMessage', [
            'chat_id' => $this->chatId, 
            'text' => $text, 
            'reply_markup' => $kb
        ]); 
    }

    private function sendMessageToId($chatId, $text) {
        return $this->req('sendMessage', [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ]);
    }

    private function editMessageText($text, $kb = null) {
        return $this->req('editMessageText', [
            'chat_id' => $this->chatId,
            'message_id' => $this->messageId,
            'text' => $text,
            'parse_mode' => 'HTML',
            'reply_markup' => $kb
        ]);
    }

    private function editMessageCaption($chatId, $messageId, $caption) {
        return $this->req('editMessageCaption', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'caption' => $caption,
            'parse_mode' => 'HTML'
        ]);
    }

    private function sendPhoto($chatId, $photo, $caption, $kb = null) {
        return $this->req('sendPhoto', [
            'chat_id' => $chatId,
            'photo' => $photo,
            'caption' => $caption,
            'reply_markup' => $kb,
            'parse_mode' => 'HTML'
        ]);
    }

    private function answerCallbackQuery($text, $alert = false) {
        return $this->req('answerCallbackQuery', [
            'callback_query_id' => $this->update['callback_query']['id'],
            'text' => $text,
            'show_alert' => $alert
        ]);
    }
    
    private function deleteMessage($msgId) {
        return $this->req('deleteMessage', [
            'chat_id' => $this->chatId,
            'message_id' => $msgId
        ]);
    }
    
private function req($method, $datas) {
        // اطمینان از اینکه حالت HTML همیشه فعال است
        if (!isset($datas['parse_mode'])) {
            $datas['parse_mode'] = 'HTML';
        }

        $ch = curl_init("https://api.telegram.org/bot" . Config::BOT_TOKEN . "/$method");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $datas);
        $res = curl_exec($ch);
        
        if (curl_errno($ch)) {
             // Logger::log("Curl Error: " . curl_error($ch));
        }
        
        curl_close($ch);
        return json_decode($res, true);
    }
    // تابع کمکی: استخراج کد واقعی VLESS/VMESS از لینک سابسکرایب
    private function getRealConfig($subUrl) {
        $ch = curl_init($subUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'v2rayNG/1.8.5'); // شبیه‌سازی کلاینت
        $response = curl_exec($ch);
        curl_close($ch);

        if (!$response) return null;

        // لینک‌های سابسکرایب معمولاً Base64 هستند
        $decoded = base64_decode($response);
        
        // اگر دیکود شد و توش vless یا vmess داشت، همونو برگردون
        if (strpos($decoded, 'vless://') !== false || strpos($decoded, 'vmess://') !== false) {
            return trim($decoded);
        }
        
        // اگر دیکود نشد (شاید خود متن پلن باشه)
        return $response;
    }
    
    // --- تابع اختصاصی ساخت لینک VMess (طبق عکس شما) ---
// --- تابع اختصاصی ساخت لینک VMess ---
// --- تابع اختصاصی ساخت لینک سابسکرایب هیدیفای ---
    private function generateSubLink($uuid, $name, $location = null) {
        $domain = \App\Config\Config::HIDDIFY_DOMAIN;
        $userPath = \App\Config\Config::HIDDIFY_USER_PATH;
        
        // ترکیب دقیق لینک: دامین + مسیر کاربر + UUID + ساب64 + یوزرنیم
        return "https://{$domain}/{$userPath}/{$uuid}/sub64/?asn=unknown#" . rawurlencode($name);
    }
    
    // نمایش فاکتور خرید (اصلاح شده و ایمن)
private function showInvoice($planId, $couponCode = null) {
        $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ? AND is_active = 1");
        $stmt->execute([$planId]);
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$plan) {
            $this->sendMessage("خطا: محصول یافت نشد.");
            return;
        }

        // 1. دریافت تخفیف نمایندگی کاربر
        $stmtUser = $this->db->prepare("SELECT discount_percent FROM users WHERE chat_id = ?");
        $stmtUser->execute([$this->chatId]);
        $resellerPercent = $stmtUser->fetchColumn() ?: 0;

        // 2. دریافت درصد کد تخفیف (اگر ارسال شده باشد)
        $couponPercent = 0;
        if ($couponCode) {
            $stmtC = $this->db->prepare("SELECT percent FROM coupons WHERE code = ?");
            $stmtC->execute([$couponCode]);
            $couponPercent = $stmtC->fetchColumn() ?: 0;
        }

        // --- محاسبات ریاضی ---
        $originalPrice = $plan['price'];
        $priceStep1 = $originalPrice;
        $msgDetails = "";

        // الف) اعمال تخفیف نمایندگی
        if ($resellerPercent > 0) {
            $resellerDiscount = ($originalPrice * $resellerPercent) / 100;
            $priceStep1 = $originalPrice - $resellerDiscount;
            $msgDetails .= "🔖 **تخفیف نمایندگی ($resellerPercent%):** -" . number_format($resellerDiscount) . " تومان\n";
        }

        // ب) اعمال کد تخفیف (روی مبلغ باقیمانده از مرحله قبل)
        $finalPrice = $priceStep1;
        if ($couponPercent > 0) {
            $couponDiscount = ($priceStep1 * $couponPercent) / 100;
            $finalPrice = $priceStep1 - $couponDiscount;
            $msgDetails .= "🎟 **کد تخفیف ($couponCode - $couponPercent%):** -" . number_format($couponDiscount) . " تومان\n";
        }

        // ساخت پیام فاکتور
        $loc = htmlspecialchars(str_replace('_', ' ', $plan['location']));
        
        $msg = "🧾 <b>فاکتور نهایی خرید</b>\n\n";
        $msg .= "🌍 سرویس: <b>$loc</b>\n";
        $msg .= "📊 حجم: <b>{$plan['volume_gb']} GB</b>\n";
        $msg .= "💰 مبلغ پایه: " . number_format($originalPrice) . " تومان\n";
        $msg .= "➖➖➖➖➖➖➖➖\n";
        $msg .= $msgDetails; // نمایش جزئیات کسر شده
        $msg .= "➖➖➖➖➖➖➖➖\n";
        $msg .= "💸 <b>مبلغ قابل پرداخت: " . number_format($finalPrice) . " تومان</b>\n\n";
        
        if (!$couponCode) {
            $msg .= "👇 اگر کد تخفیف دارید، از دکمه زیر استفاده کنید:";
        }

        $backData = "dur_" . $plan['location'] . "|" . $plan['duration_title'];
        
        // ارسال به کیبورد (اگر کد داشت، دکمه حذف میاد، اگر نداشت دکمه ثبت میاد)
        $this->sendMessage($msg, Keyboard::invoiceButtons($plan['id'], $finalPrice, $backData, !empty($couponCode)));
    }
    
    // تابع کمکی برای ایمن‌سازی متن‌ها در حالت HTML
    private function escape($text) {
        return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
    }
    // ==========================================
    //           توابع مدیریت سرویس و استعلام
    // ==========================================

// ==========================================
    // نمایش جزئیات سرویس خریداری شده (نسخه هیدیفای)
    // ==========================================
    private function showServiceDetails($orderId) {
        $stmt = $this->db->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$orderId]);
        $order = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$order) {
            $this->answerCallbackQuery("❌ سرویس یافت نشد!", true);
            return;
        }

        $this->answerCallbackQuery("🔄 در حال دریافت اطلاعات از سرور...");

        // دریافت آمار زنده از هیدیفای
        $stats = $this->getRealTimeStats($order['uuid']); 
        
        if ($stats) {
            // دریافت نام کاربری واقعی از پنل هیدیفای
            $panel = new \App\Helpers\V2RayPanel();
            $userInfo = $panel->getUserStats($order['uuid']);
            $realName = $userInfo ? $userInfo['name'] : "User_" . $this->chatId;

            // تولید لینک سابسکرایب بدون نیاز به لوکیشن
            $link = $this->generateSubLink($order['uuid'], $realName);
            
            // تعیین وضعیت با ایموجی
            $statusEmoji = ($stats['expiry'] == 'منقضی شده' || $stats['remain'] == '0 B') ? '🔴 غیرفعال' : '🟢 فعال';
            
            $msg = "📊 <b>وضعیت سرویس شما:</b>\n\n";
            $msg .= "🔰 <b>وضعیت:</b> $statusEmoji\n";
            $msg .= "👤 <b>نام کاربری:</b> <code>$realName</code>\n";
            $msg .= "➖➖➖➖➖➖➖➖\n";
            $msg .= "🔋 <b>حجم کل:</b> {$stats['total']}\n";
            $msg .= "🔻 <b>مصرف شده:</b> {$stats['used']}\n";
            $msg .= "✅ <b>باقی‌مانده:</b> {$stats['remain']}\n";
            $msg .= "➖➖➖➖➖➖➖➖\n";
            $msg .= "📅 <b>تاریخ انقضا:</b> {$stats['expiry']}\n\n";
            $msg .= "🔌 <b>لینک اتصال اختصاصی شما:</b>\n\n";
            $msg .= "<code>$link</code>";
            
            $kb = json_encode(['inline_keyboard' => [[
                ['text' => '🔄 بروزرسانی آمار', 'callback_data' => "my_service_$orderId"],
                ['text' => '🔙 بازگشت', 'callback_data' => 'back_to_main_menu']
            ]]]);

            $this->editMessageText($msg, $kb);
        } else {
            $kb = json_encode(['inline_keyboard' => [[
                ['text' => '🔙 بازگشت', 'callback_data' => 'back_to_main_menu']
            ]]]);
            $this->editMessageText("⚠️ <b>سرویس یافت نشد!</b>\n\nاطلاعات این سرویس در سرور وجود ندارد (احتمالاً به دلیل اتمام زمان/حجم حذف شده است).", $kb);
        }
    }
    

// ==========================================
    // پردازش استعلام (اصلاح شده و نهایی)
    // ==========================================
// ==========================================
    // پردازش استعلام (نسخه بدون ستاره)
    // ==========================================
    private function processInquiry($text) {
        // 1. استخراج UUID از متن ارسالی (لینک یا کد)
        $uuid = $this->extractUuid($text);

        if (!$uuid) {
            $this->sendMessage("❌ <b>فرمت نامعتبر است.</b>\n\nلطفاً لینک کانفیگ (vmess/vless) یا کد UUID را به صورت صحیح ارسال کنید.");
            return;
        }

        $this->sendMessage("⏳ <b>در حال دریافت اطلاعات از سرور...</b>");

        // 2. دریافت آمار از پنل 
        $stats = $this->getRealTimeStats($uuid);

        if ($stats) {
            // تعیین وضعیت با ایموجی
            $statusEmoji = ($stats['expiry'] == 'منقضی شده' || $stats['remain'] == '0 B') ? '🔴 غیرفعال' : '🟢 فعال';

            $msg = "📊 <b>نتیجه استعلام سرویس:</b>\n\n";
            $msg .= "🔰 <b>وضعیت:</b> $statusEmoji\n";
            $msg .= "🆔 <b>شناسه:</b> <code>$uuid</code>\n";
            $msg .= "➖➖➖➖➖➖➖➖\n";
            $msg .= "🔋 <b>حجم کل:</b> {$stats['total']}\n";
            $msg .= "🔻 <b>مصرف شده:</b> {$stats['used']}\n";
            $msg .= "📦 <b>باقی‌مانده:</b> {$stats['remain']}\n";
            $msg .= "➖➖➖➖➖➖➖➖\n";
            $msg .= "📅 <b>تاریخ انقضا:</b> {$stats['expiry']}\n";
            
            // دکمه بازگشت
            $this->sendMessage($msg, Keyboard::mainMenu($this->isAdmin));
            
            // پاک کردن مرحله کاربر (برای اینکه دوباره مجبور نباشد استعلام بزند)
            $this->updateUserStep('none');
            
        } else {
            $this->sendMessage("❌ <b>سرویس یافت نشد.</b>\n\nممکن است سرویس حذف شده باشد یا مربوط به این سرور نباشد.", Keyboard::mainMenu($this->isAdmin));
            $this->updateUserStep('none');
        }
    }
    
    // تابع کمکی برای کشیدن نوار درصد
    private function drawProgressBar($percent) {
        $fill = round($percent / 10);
        return str_repeat("🟥", $fill) . str_repeat("⬜️", 10 - $fill);
    }
    // تابع کمکی برای ساخت نوار پیشرفت
    private function makeProgressBar($percent) {
        $totalBlocks = 10;
        $filledBlocks = round(($percent / 100) * 100 / 10); // مقیاس 10 تایی
        $emptyBlocks = $totalBlocks - $filledBlocks;
        
        return str_repeat("🟥", $filledBlocks) . str_repeat("⬜️", $emptyBlocks);
    }

// ==========================================
    // دریافت آمار واقعی از پنل هیدیفای (Hiddify API)
    // ==========================================
    private function getRealTimeStats($uuid) {
        $panel = new \App\Helpers\V2RayPanel();
        $userInfo = $panel->getUserStats($uuid);

        if (!$userInfo) return null;

        // استخراج اطلاعات از API هیدیفای
        $totalGB = $userInfo['usage_limit_GB'] ?? 0;
        $usedGB = $userInfo['current_usage_GB'] ?? 0;
        
        $totalStr = ($totalGB > 0) ? $totalGB . " GB" : "نامحدود ∞";
        $usedStr = round($usedGB, 2) . " GB";
        
        if ($totalGB > 0) {
            $remainGB = max(0, $totalGB - $usedGB);
            $remainStr = round($remainGB, 2) . " GB";
        } else {
            $remainStr = "نامحدود ∞";
        }

        // محاسبه تاریخ انقضا در هیدیفای
        $days = $userInfo['package_days'] ?? 0;
        $startDate = $userInfo['start_date'] ?? null; 
        
        if ($days == 0) {
            $expiryStr = "نامحدود ∞";
        } elseif (!$startDate || $startDate == "None" || trim($startDate) == "") {
            $expiryStr = "پس از اولین اتصال ($days روز)";
        } else {
            // تبدیل تاریخ هیدیفای
            $expiryTime = strtotime($startDate) + ($days * 86400); 
            if (time() > $expiryTime) {
                $expiryStr = "منقضی شده";
            } else {
                $dt = new \DateTime("@" . $expiryTime);
                $dt->setTimezone(new \DateTimeZone('Asia/Tehran'));
                $expiryStr = $dt->format("Y/m/d"); 
            }
        }

        return [
            'total'  => $totalStr,
            'used'   => $usedStr,
            'remain' => $remainStr,
            'expiry' => $expiryStr
        ];
    }
    // ==========================================
    // استخراج هوشمند UUID از انواع لینک‌ها
    // ==========================================
// ==========================================
    // استخراج هوشمند UUID از لینک سابسکرایب هیدیفای
    // ==========================================
    private function extractUuid($string) {
        $string = trim($string);
        // این ریجکس فوق‌العاده است و فقط شناسه 36 کاراکتری (UUID) را از هر لینکی بیرون می‌کشد
        if (preg_match('/[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}/i', $string, $matches)) {
            return $matches[0];
        }
        return null;
    }
    // 5. فرمت کردن بایت به گیگابایت/مگابایت
    private function formatBytes($bytes, $precision = 2) {
        if ($bytes <= 0) return '0 B';
        $base = log($bytes, 1024);
        $suffixes = array('B', 'KB', 'MB', 'GB', 'TB');
        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffixes[floor($base)];
    }
    
    // ==========================================
    //      توابع عمومی مخصوص ادمین (Public)
    // ==========================================

    // تابع کپی برای ارسال همگانی (جدید)
    public function copyMessage($chatId, $fromChatId, $messageId) {
        return $this->req('copyMessage', [
            'chat_id' => $chatId,
            'from_chat_id' => $fromChatId,
            'message_id' => $messageId
        ]);
    }

    // تابع فوروارد برای ارسال همگانی (جدید)
    public function forwardMessage($chatId, $fromChatId, $messageId) {
        return $this->req('forwardMessage', [
            'chat_id' => $chatId,
            'from_chat_id' => $fromChatId,
            'message_id' => $messageId
        ]);
    }

    // ✅ تابع جدید برای ویرایش پیام (مخصوص ادمین)
    // نامش را تغییر دادیم به editMessageByAdmin تا با تابع قبلی قاطی نشود
    public function editMessageByAdmin($chatId, $messageId, $text) {
        return $this->req('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ]);
    }
    
    // ✅ تابع جدید برای حذف پیام (مخصوص ادمین)
    // نامش را تغییر دادیم به deleteMessageByAdmin
    public function deleteMessageByAdmin($chatId, $messageId) {
        return $this->req('deleteMessage', [
            'chat_id' => $chatId,
            'message_id' => $messageId
        ]);
    }

// تابع هوشمند ساخت لینک بر اساس نوع پلن
private function generateConfigByPlan($uuid, $username, $location) {
        $domain = "server.wzxshop.ir"; // دامین سرور شما
        
        // حالت ۱: اگر پلن "مولتی لوکیشن" باشد
        if ($location == \App\Config\Config::MULTI_LOC_NAME) {
            $configs = [];
            
            // حلقه روی تمام لوکیشن‌های تعریف شده در Config.php
            foreach (\App\Config\Config::LOCATIONS as $locName => $details) {
                // ساخت نام نمایشی (مثلا: User_123_France_🇫🇷)
                // از rawurlencode استفاده میکنیم تا ایموجی‌ها لینک را خراب نکنند
                $remark = rawurlencode($username . "_" . $locName);
                
                $configs[] = "vless://$uuid@$domain:{$details['port']}?encryption=none&type=tcp&headerType=http&path=%2F&host={$details['host']}#$remark";
            }
            // همه لینک‌ها را با دو اینتر جدا میکنیم
            return implode("\n\n", $configs);
        }
        
        // حالت ۲: اگر پلن "تک لوکیشن" باشد
        // چک میکنیم آیا این لوکیشن در لیست ما وجود دارد؟
        if (isset(\App\Config\Config::LOCATIONS[$location])) {
            $details = \App\Config\Config::LOCATIONS[$location];
            $remark = rawurlencode($username . "_" . $location);
            
            return "vless://$uuid@$domain:{$details['port']}?encryption=none&type=tcp&headerType=http&path=%2F&host={$details['host']}#$remark";
        }
        
        // حالت اضطراری (اگر لوکیشن پیدا نشد)
        return "❌ خطا: لوکیشن تنظیم شده برای این پلن معتبر نیست.";
    }    
    
    private function checkAndApplyCoupon($user, $code) {
        $planId = $user['temp_data']; // آیدی پلن که در دکمه ذخیره کردیم

        // جستجوی کد در دیتابیس
        $stmt = $this->db->prepare("SELECT * FROM coupons WHERE code = ?");
        $stmt->execute([$code]);
        $coupon = $stmt->fetch(PDO::FETCH_ASSOC);

        // بررسی اعتبار
        if (!$coupon) {
            $this->sendMessage("⛔️ کد تخفیف نامعتبر است.\nلطفاً مجدد تلاش کنید یا انصراف دهید.");
            return;
        }

        // بررسی ظرفیت
        if ($coupon['used_count'] >= $coupon['max_uses']) {
            $this->sendMessage("⛔️ متاسفانه ظرفیت استفاده از این کد تکمیل شده است.");
            return;
        }

        // اگر معتبر بود
        $this->updateUserStep('coupon_applied', $planId . "|" . $code); // ذخیره وضعیت: پلن|کد
        $this->sendMessage("✅ **کد تخفیف اعمال شد!**");
        
        // نمایش مجدد فاکتور با کد
        $this->showInvoice($planId, $code);
    }
    
    // ==========================================
    //            بررسی قفل کانال
    // ==========================================
    private function checkMandatoryJoin() {
        $stmt = $this->db->query("SELECT * FROM channels");
        $channels = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        
        if (empty($channels)) return true; // اگر کانالی ثبت نشده بود، آزاد است
        
        $notJoined = false;
        foreach ($channels as $channel) {
            $res = $this->req('getChatMember', [
                'chat_id' => $channel['chat_id'],
                'user_id' => $this->chatId
            ]);
            
            // وضعیت‌های عدم عضویت: left (خارج شده) یا kicked (بن شده)
            $status = $res['result']['status'] ?? 'left';
            if ($status == 'left' || $status == 'kicked') {
                $notJoined = true;
                break; // اگر حتی یک کانال را عضو نباشد، قفل عمل می‌کند
            }
        }
        
        if ($notJoined) {
            return $channels; // لیست کانال‌ها را برمی‌گردانیم تا کیبوردش ساخته شود
        }
        
        return true; // در تمام کانال‌ها عضو است
    }
    
    // ==========================================
    //            سیستم پاداش زیرمجموعه‌گیری
    // ==========================================
    private function giveReferralReward($buyerChatId, $purchaseAmount) {
        // پیدا کردن معرفِ این خریدار
        $stmt = $this->db->prepare("SELECT invited_by, full_name FROM users WHERE chat_id = ?");
        $stmt->execute([$buyerChatId]);
        $buyer = $stmt->fetch(\PDO::FETCH_ASSOC);
        
        if ($buyer && !empty($buyer['invited_by'])) {
            $inviterId = $buyer['invited_by'];
            
            // محاسبه 5 درصد پورسانت
            $reward = ($purchaseAmount * 5) / 100;
            
            if ($reward > 0) {
                // واریز به کیف پول معرف
                $this->db->prepare("UPDATE users SET wallet = wallet + ? WHERE chat_id = ?")->execute([$reward, $inviterId]);
                
                // ارسال پیام تبریک به معرف
                $buyerName = $buyer['full_name'] ?: "یک کاربر";
                $msg = "🎉 <b>تبریک!</b>\n\n";
                $msg .= "زیرمجموعه شما ($buyerName) یک سرویس خریداری کرد!\n";
                $msg .= "مبلغ <b>" . number_format($reward) . " تومان</b> (5% پورسانت) به کیف پول شما اضافه شد. 💰";                
                $this->sendMessageToId($inviterId, $msg);
            }
        }
    }
    
} // پایان کلاس BotController